/*++

	Copyright (c) 1998,  Dr. Johannes Heidenhain GmbH

	Module Name:	Exports.c
	
	Abstract:		Windows DLL for IK 121

	Notes:			This DLL provides IK 121 card functions for IK 121.
					IK 121 hardware is accessed through IK121Drv.Sys.



	History:		-	IKDefine command added				Schroll E.		11.06.99	

					-   IKVersion command added				Schroll E.		11.06.99

--*/


#include <math.h>
#include <windows.h>
#include "dllglob.h"
#include "..\Include\DrvFunc.h"




#ifdef _Windows95
  #ifdef _DEBUG
	const char DLLVers[15] = "DLL95 V1.1(D)";
  #else
	const char DLLVers[15] = "DLL95 V1.1(R)";
  #endif
#else
  #ifdef _DEBUG
	const char DLLVers[15] = "DLLNT V1.1(D)";
  #else
	const char DLLVers[15] = "DLLNT V1.1(R)";
  #endif				
#endif

const char Cnt_G26[15]		= "G26";
const char Cnt_G38[15]		= "G38";
const char Cnt_unknown[15]	= "???";




// Global variable definitions
USHORT	RegW;					// Register value word
ULONG	RegL;					// Register value double-word
BOOL	RefStatus;				// Status of REF

USHORT	CodRefLat [16];			// Latch for coded REF
ULONG	CodRefDist[16];			// Constant distance between coded reference marks
USHORT	CodRefSta [16];			// Status for coded reference run

USHORT Ram_OfsReg0 [16];		// Shadow register for offset  0 degree (register 16H/17H)
USHORT Ram_OfsReg90[16];		// Shadow register for offset 90 degree (register 18H/19H)
USHORT Ram_LatchReg[16];		// Shadow register for latch register   (register 12H/13H)


double	FirstCRef[16];			// Value first reference mark during coded reference run


#ifdef _Windows95
	#include <conio.h>

	// Global variable definitions for Windows 95
	ULONG PortBase[8];

#else
	#include <winioctl.h>

	// Global variable definitions for Windows NT
	HANDLE	m_hIK121Drv;		// Handle for device driver
#endif


#ifdef _Windows95


////////////////////////////////////////////////////////////////////
// Windows 95
// IKDefine: Sets port address(es) of installed card(s) into pBuffer
////////////////////////////////////////////////////////////////////

DLLEXPORT BOOL WINAPI IKDefine (ULONG* pBuffer8)
{
	USHORT Ax;
	
	if (!pBuffer8) return FALSE;
	
	for (Ax=0; Ax<8; Ax++)
		PortBase[Ax] = *(pBuffer8+Ax);
	return TRUE;
}



///////////////////////////////////////////////////////////////////
// Windows 95
// IKFind: Returns port address(es) of installed card(s) in pBuffer
///////////////////////////////////////////////////////////////////

DLLEXPORT BOOL WINAPI IKFind (ULONG* pBuffer8)
{
	USHORT Ax;
	
	if (!pBuffer8) return FALSE;
	
	for (Ax=0; Ax<8; Ax++)
	{
		if (PortBase[Ax])
		{
			if (!IKInputW (Ax, StatReg3, &RegW)) return FALSE;
			if ( (RegW & 0xFF00) == 0x0800 || (RegW & 0xFF00) == 0x0900 )
				*(pBuffer8+Ax)=PortBase[Ax];
			else 
				*(pBuffer8+Ax)=0;
		}
		else
			*(pBuffer8+Ax)=0;
	}
	return TRUE;
}



//////////////////////////////////////////////////////////
// Windows 95
// IKInputW: Returns word read from Axis at Adr to pBuffer
//////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKInputW (USHORT Axis, USHORT Adr, USHORT* pBuffer)
{
	USHORT cardNum =  Axis >> 1;				// Axis number --> Card number (2 axes on one card)
	USHORT adrSeg  = (Adr  & 0x1C) >> 2;		// Reg  number --> Page number
	USHORT adrOfs  = (Axis & 0x01) << 2;		// Axis number --> Axis select
	USHORT adrSel  =  Adr  & 0x03;				// Reg  number --> Reg  select

	if (Axis>15)			return FALSE;
	if (!PortBase[Axis>>1]) return FALSE;
	if (Adr >0x1E)			return FALSE;
	if (!pBuffer)			return FALSE; 
	
	_outpw ( (USHORT)(PortBase[cardNum] | 8), adrSeg );
	*pBuffer = _inpw  ( (USHORT)(PortBase[cardNum] | adrOfs | adrSel) );

	return TRUE;
}


/////////////////////////////////////////////////////////////////
// Windows 95
// IKInputL: Returns double-word read from Axis at Adr to pBuffer
/////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKInputL (USHORT Axis, USHORT Adr, ULONG* pBuffer)
{
	USHORT cardNum =  Axis >> 1;				// Axis number --> Card number (2 axes on one card)
	USHORT adrSeg  = (Adr  & 0x1C) >> 2;		// Reg  number --> Page number
	USHORT adrOfs  = (Axis & 0x01) << 2;		// Axis number --> Axis select
	USHORT adrSel  =  Adr  & 0x03;				// Reg  number --> Reg  select

	PUSHORT pBufferW = (PUSHORT) pBuffer;

	if (Axis>15)			return FALSE;
	if (!PortBase[Axis>>1]) return FALSE;
	if (Adr >0x1C)			return FALSE;
	if (!pBuffer)			return FALSE; 

	_outpw ( (USHORT)(PortBase[cardNum] | 8), adrSeg );

	if (adrSel==0)		// Read longword from quad address --> same page
		*pBuffer = _inpd  ( (USHORT)(PortBase[cardNum] | adrOfs | adrSel) );
	else				// Read longword from different pages 
	{
		*pBufferW = _inpw  ( (USHORT)(PortBase[cardNum] | adrOfs | adrSel) );

		adrSeg=(adrSeg+1) & 0x07; 	// Next page number !!!
		adrSel=(adrSel+2) & 0x03; 	// Next register address !!!
		_outpw ( (USHORT)(PortBase[cardNum] | 8), adrSeg );
		*(pBufferW+1) = _inpw  ( (USHORT)(PortBase[cardNum] | adrOfs | adrSel) );
	}

	return TRUE;
}


/////////////////////////////////////////////////
// Windows 95
// IKOutput: Writes word from data to Axis at Adr
/////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKOutput (USHORT Axis, USHORT Adr, USHORT Data)
{
	USHORT cardNum=  Axis >> 1;				// Axis number --> Card number (2 axes on one card)
	USHORT adrSeg = (Adr  & 0x1C) >> 2;		// Reg  number --> Page number
	USHORT adrOfs = (Axis & 0x01) << 2;		// Axis number --> Axis select
	USHORT adrSel =  Adr  & 0x03;			// Reg  number --> Reg  select

	if (Axis>15)			return FALSE;
	if (!PortBase[Axis>>1]) return FALSE;
	if (Adr >0x1E)			return FALSE;


	_outpw ( (USHORT)(PortBase[cardNum] | 8)              , adrSeg );
	_outpw ( (USHORT)(PortBase[cardNum] | adrOfs | adrSel), Data   );

	if (Adr==OfsReg0)  Ram_OfsReg0 [Axis]=Data;				// Store Ofs0  value in shadow register
	if (Adr==OfsReg90) Ram_OfsReg90[Axis]=Data;				// Store Ofs90 value in shadow register
	if (Adr==LatchReg) Ram_LatchReg[Axis]=Data & 0x77FF;	// Store latch register value in shadow register

	return TRUE;
}




void delay_us(USHORT us)
{
	BYTE lo, hi;
	long summ,c,d,e,max;

	max=us*2380L/1000L;
	summ=0L;
	_outp(67,0);					
	lo=_inp(64); 					// Read low  byte
	hi=_inp(64); 					// Read high byte
	d=(hi<<8)+lo;					// Start time
	while(summ<max)
	{
		_outp(67,0);
		lo=_inp(64);  				// Read low  byte
		hi=_inp(64);  				// Read high byte
	    c=(hi<<8)+lo; 				// Current time
		e=d-c;						// Elapsed time
		if(e<0L) e=e+65535L;		// If negative correct overflow
		summ=summ+e;	
		d=c;
	}
}




///////////////////////////////////////////////////////////////////
// Windows 95
// IKSetI2C: Writes word Data to Latch Register (I2C clock and data)
////////////////////////////////////////////////////////////////////

BOOL SetI2C (USHORT Card, USHORT Data)
{
	// Address SCL/SDA port = 0x12 !!
	USHORT adrSeg = 0x04;	// Page number I2C-Input
	USHORT adrOfs = 0x04;	// Axis 2
	USHORT adrSel = 0x02;	// Reg  select 2

	if (Card>7)			 return FALSE;
	if (!PortBase[Card]) return FALSE;

	_outpw ( (USHORT)(PortBase[Card] | 8)              , adrSeg );
	_outpw ( (USHORT)(PortBase[Card] | adrOfs | adrSel), Data   );

	delay_us (10);
	return TRUE;
}



/////////////////////////////////////////////////////
// Windows95
// GetVers: Returns software version of device driver
/////////////////////////////////////////////////////
BOOL GetVers (USHORT Axis, char* pVersDrv)
{
		*pVersDrv++  = '-';
		*pVersDrv++  = '-';
		*pVersDrv++  = '-';
		*pVersDrv++  = 0x00;

		return TRUE;
}





#else		// ifdef _Windows95




////////////////////////////////////////////////////////////////////
// Windows NT	( !!! Function not available under Windows NT !!! )
// IKDefine: Sets port address(es) of installed card(s) into pBuffer
////////////////////////////////////////////////////////////////////

DLLEXPORT BOOL WINAPI IKDefine (ULONG* pBuffer8)
{
	return FALSE;
}



///////////////////////////////////////////////////////////////////
// Windows NT
// IKFind: Returns port address(es) of installed card(s) in pBuffer
///////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKFind ( ULONG* pBuffer8)
{
	DWORD  byteCount;

	if (!pBuffer8) return FALSE; 

	if  (!DeviceIoControl	( m_hIK121Drv,				// Handle to device driver
							  (DWORD)IOCTL_STATUS,		// I/O control code
							  pBuffer8,					// Pointer to input buffer
							  8*4,						// Input buffer size
							  pBuffer8,					// Pointer to output buffer
							  8*4,						// Output buffer size
							  &byteCount,				// Bytes returned
							  NULL ))					// Not overlapped
		return (FALSE);
	return (byteCount ? TRUE : FALSE);
}



//////////////////////////////////////////////////////////
// Windows NT
// IKInputW: Returns word read from Axis at Adr to pBuffer
//////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKInputW (USHORT Axis, USHORT Adr, USHORT* pBuffer)
{
	struct p { USHORT axis, adr, data; } port;
 	DWORD  byteCount;
   
	if (Axis>15)   return FALSE;
	if (Adr >0x1E) return FALSE;
	if (!pBuffer)  return FALSE; 

	port.axis = Axis; port.adr  = Adr;
	if  (!DeviceIoControl	( m_hIK121Drv,				// Handle to device driver
							  (DWORD)IOCTL_INPUT,		// I/O control code
							  &port,					// Pointer to input buffer
							  (DWORD) sizeof(port),		// Input buffer size
							  pBuffer,					// Pointer to output buffer
							  (DWORD) sizeof(*pBuffer),	// Output buffer size
							  &byteCount,				// Bytes returned
							  NULL ))					// Not overlapped
		return (FALSE);
	return (byteCount ? TRUE : FALSE);
}


/////////////////////////////////////////////////////////////////
// Windows NT
// IKInputL: Returns double-word read from Axis at Adr to pBuffer
/////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKInputL (USHORT Axis, USHORT Adr, ULONG* pBuffer)
{
	struct p { USHORT axis, adr, data; } port;
	DWORD  byteCount;

	if (Axis>15)   return FALSE;
	if (Adr >0x1C) return FALSE;
	if (!pBuffer)  return FALSE; 

	port.axis = Axis; port.adr  = Adr;
	if  (!DeviceIoControl	( m_hIK121Drv,				// Handle to device driver
							  (DWORD)IOCTL_INPUT,		// I/O control code
							  &port,					// Pointer to input buffer
							  (DWORD) sizeof(port),		// Input buffer size
							  pBuffer,					// Pointer to output buffer
							  (DWORD) sizeof(*pBuffer),	// Output buffer size
							  &byteCount,				// Bytes returned
							  NULL ))					// Not overlapped
		return (FALSE);
	return (byteCount ? TRUE : FALSE);
}


/////////////////////////////////////////////////
// Windows NT
// IKOutput: Writes word Data to Axis at Adr
/////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKOutput (USHORT Axis, USHORT Adr, USHORT Data)
{
	struct p { USHORT axis, adr, data; } port;
	DWORD  byteCount;

	if (Axis>15)   return FALSE;
	if (Adr >0x1E) return FALSE;

	port.axis = Axis; port.adr = Adr; port.data = Data;
	if  (!DeviceIoControl	( m_hIK121Drv,				// Handle to device driver
							  (DWORD)IOCTL_OUTPUT,		// I/O control code
							  &port,					// Pointer to output buffer
							  (DWORD) sizeof(port),		// Output buffer size
							  &port,					// Pointer to output buffer
							  (DWORD) sizeof(port),		// Output buffer size
							  &byteCount,				// Bytes returned
							  NULL ))					// Not overlapped
		return (FALSE);
 
	if (Adr==OfsReg0)  Ram_OfsReg0 [Axis]=Data;				// Store Ofs0  value in shadow register
	if (Adr==OfsReg90) Ram_OfsReg90[Axis]=Data;				// Store Ofs90 value in shadow register
	if (Adr==LatchReg) Ram_LatchReg[Axis]=Data & 0x77FF;	// Store latch register value in shadow register

	return (byteCount ? TRUE : FALSE);
}


///////////////////////////////////////////////////////////////////
// Windows NT
// IKSetI2C: Writes word Data to Latch Register (I2C clock and data)
////////////////////////////////////////////////////////////////////

BOOL SetI2C (USHORT Card, USHORT Data)
{
	struct p { USHORT card, data; } port;
	DWORD  byteCount;

	if (Card>7) return FALSE;

	port.card = Card; port.data = Data;
	if  (!DeviceIoControl	( m_hIK121Drv,				// Handle to device driver
							  (DWORD)IOCTL_SETI2C,		// I/O control code
							  &port,					// Pointer to output buffer
							  (DWORD) sizeof(port),		// Output buffer size
							  &port,					// Pointer to output buffer
							  (DWORD) sizeof(port),		// Output buffer size
							  &byteCount,				// Bytes returned
							  NULL ))					// Not overlapped
		return (FALSE);
	return (byteCount ? TRUE : FALSE);
}



/////////////////////////////////////////////////////
// WindowsNT
// GetVers: Returns software version of device driver
/////////////////////////////////////////////////////
BOOL GetVers (USHORT Axis, char* pVersDrv)
{
	DWORD  byteCount;
	USHORT i;
	char Buffer[25];

	if  (!DeviceIoControl	( m_hIK121Drv,				// Handle to device driver
							  (DWORD)IOCTL_VERSION,		// I/O control code
							  NULL,						// Pointer to input buffer
							  (DWORD)sizeof(0),			// Input buffer size
							  &Buffer,					// Pointer to output buffer
							  (DWORD) sizeof(Buffer),	// Output buffer size
							  &byteCount,				// Bytes returned
							  NULL ))					// Not overlapped

		return FALSE;

	if (byteCount)
	{
		i=0;
		do {
			*pVersDrv++=Buffer[i];
		} while (Buffer[i++]!=0x00);

		return TRUE;
	}
	else
		return FALSE;
}



#endif		// ifdef _Windows95





///////////////////////////
// IKInit: Initializes Axis
///////////////////////////
DLLEXPORT BOOL WINAPI IKInit (USHORT Axis, USHORT Mode)
{
	SHORT Ofs0, Ofs90;


	if (!IKOutput(Axis, ContrReg3, 0x0000))		return FALSE;		// 16-bit-bus / clock <= 16MHz

    RegW = 0x0001;													// Interpolation mode
	if (Mode & 0x0001) RegW=RegW | 0x0040;							// 32/48-bit counter
	if (!IKOutput(Axis, InitReg1, RegW))		return FALSE;            

	if (!IKOutput(Axis, LatchReg, 0x0000))		return FALSE;		// Clear latch register

	if (!IKOutput(Axis, IntrReg, 0x0000))		return FALSE;		// Clear interrupt enable register

	if (!IKLoadOffset  (Axis, &Ofs0, &Ofs90))	return FALSE;		// Read offset values from pot register
	if (!IKWriteOffset (Axis,  Ofs0,  Ofs90))	return FALSE;		// Write offset values of axis

	if (!IKOutput(Axis, StatReg3, cmdADFree))	return FALSE;		// Unfreeze value



	if (!IKOutput(Axis, ContrReg, 0x00F0))		return FALSE;		// Counter stop
																	// Counter reset
																	// Clear frequency error
																	// Clear amplitude error

	if (!IKOutput(Axis, RefCmdReg, 0x0000))		return FALSE;		// REF without function

    if (!IKInputL(Axis, DataReg0,   &RegL))		return FALSE;		// Read latch register 0
    if (!IKInputW(Axis, DataReg0+4, &RegW))		return FALSE;

    if (!IKInputL(Axis, DataReg1,   &RegL))		return FALSE;		// Read latch register 1
    if (!IKInputW(Axis, DataReg1+4, &RegW))		return FALSE;

    if (!IKOutput(Axis, ContrReg, 0x0003))		return FALSE;		// Clear latch register

    if (!IKInputL(Axis, DataReg0,   &RegL))		return FALSE;		// Read latch register 0
    if (!IKInputW(Axis, DataReg0+4, &RegW))		return FALSE;

    if (!IKInputL(Axis, DataReg1,   &RegL))		return FALSE;		// Read latch register 1
    if (!IKInputW(Axis, DataReg1+4, &RegW))		return FALSE;

	return IKOutput(Axis, ContrReg, cmdStart);						// Counter start
}



/////////////////////////////////////////////////////
// IKVersion: Returns version of card, driver and dll
/////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKVersion (USHORT Axis, char* pVersCard, char* pVersDrv, char* pVersDll)
{

	USHORT  i;
	const char*	pVersCounter;

    if (!IKInputW(Axis, StatReg3, &RegW))	 return FALSE;

	switch (RegW & 0xFF00)
	{
		case 0x0800:	pVersCounter = &Cnt_G26[0];
						break;

		case 0x0900:	pVersCounter = &Cnt_G38[0];
						break;

		default:		pVersCounter = &Cnt_unknown[0];
						break;
	}

	do
	{
		*pVersCard++=*pVersCounter++;							// Version of card
	} while (*pVersCounter);


	for (i=0; i<sizeof(DLLVers); i++) *pVersDll++=DLLVers[i];	// Version of DLL
	
	if (!GetVers (Axis, pVersDrv))  return FALSE;				// Get version of device driver
	for (i=0; i<sizeof(DLLVers); i++) *pVersDll++=DLLVers[i];	// Version of DLL

	return TRUE;
}


///////////////////////
// IKReset: Resets Axis
///////////////////////
DLLEXPORT BOOL WINAPI IKReset (USHORT Axis)
{
	return IKOutput(Axis, ContrReg, cmdReset);
}

///////////////////////
// IKStart: Starts Axis
///////////////////////
DLLEXPORT BOOL WINAPI IKStart (USHORT Axis)
{
	return IKOutput(Axis, ContrReg, cmdStart);
}

/////////////////////
// IKStop: Stops Axis
/////////////////////
DLLEXPORT BOOL WINAPI IKStop (USHORT Axis)
{
	return IKOutput(Axis, ContrReg, cmdStop);
}


//////////////////////////////////////////////////
// IKResetREF: Resets Axis with next reference mark
//////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKResetREF (USHORT Axis)
{
	return IKOutput(Axis, RefCmdReg, cmdResetREF);
}

//////////////////////////////////////////////////
// IKStartREF: Starts Axis with next reference mark
//////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKStartREF (USHORT Axis)
{
	return IKOutput(Axis, RefCmdReg, cmdStartREF);
}

////////////////////////////////////////////////
// IKStopREF: Stops Axis with next reference mark
////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKStopREF (USHORT Axis)
{
	return IKOutput(Axis, RefCmdReg, cmdStopREF);
}


///////////////////////////////////////////////////
// IKLatch: Stores count value of Axis in Latch 0/1
///////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKLatch (USHORT Axis, USHORT Latch)
{
	if (Axis>15) return FALSE;
	if (Latch>1) return FALSE;

	if (CodRefSta[Axis]!=0)
		if (CodRefLat[Axis]==Latch) return FALSE;		// Latch reserved for coded REF

	return IKOutput ( Axis, ContrReg, (USHORT)(cmdLatch0<<Latch) );
}


///////////////////////////////////////////////////////////////////////////////
// IKLatchREF: Stores count value of Axis in Latch 0/1 with next reference mark
///////////////////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKLatchREF (USHORT Axis, USHORT Latch)
{
	if (Axis>15) return FALSE;
	if (Latch>1) return FALSE;

	if (!IKInputW(Axis,InitReg1,&RegW)) return FALSE;
	if (Latch==0) 
	{
		if (!IKOutput(Axis,InitReg1,(USHORT)(RegW | cmdREF2L0))) return FALSE;
	}
	else
	{
		if (!IKOutput(Axis,InitReg1,(USHORT)(RegW | cmdREF2L1))) return FALSE;
	}

	if (!IKInputW(Axis,RefCmdReg,&RegW)) return FALSE;
	return IKOutput(Axis, RefCmdReg, (USHORT)(RegW & 0x003F | cmdLatchREF));
}


//////////////////////////////////////////////////////////////////
// IKLatched: Checks if count value of Axis is stored in Latch 0/1
//////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKLatched (USHORT Axis, USHORT Latch, BOOL* pStatus)
{
	*pStatus = FALSE;
	if (Latch>1) return FALSE;

	if (!IKInputW ( Axis, ContrReg, &RegW )) return FALSE;

	if ( (RegW & (cmdLatch0<<Latch)) ) *pStatus = TRUE;
	return TRUE;
}


/////////////////////////////////////////////////////////////////////
// IKWaitLatch: Wait until count value of Axis is stored in Latch 0/1
/////////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKWaitLatch (USHORT Axis, USHORT Latch)
{
	if (Latch>1) return FALSE;
	do
	{
		if (!IKInputW ( Axis, ContrReg, &RegW )) return FALSE;
	} while ( !(RegW & (cmdLatch0<<Latch)) );
	return TRUE;
}


/////////////////////////////////////////////////////////////////////
// IKStrtCodRef: Starts REF with coded reference marks with Latch 0/1
/////////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKStrtCodRef (USHORT Axis, USHORT Latch, ULONG RefDist)
{
	if (Axis>15) return FALSE;
	if (Latch>1) return FALSE;
	if (CodRefSta[Axis]!=0) return FALSE;			// Coded REF already started


	if (!IKLatchREF(Axis, Latch)) return FALSE;		// Latch with next ref. mark
	CodRefLat [Axis] = Latch;						// Latch for coded REF
	CodRefDist[Axis] = RefDist;						// Constant distance of coded ref. marks (500,1000,2000,5000)
	CodRefSta [Axis] = 1;
	return TRUE;
}


//////////////////////////////////////////////////////////////////
// IKLatched: Checks if count value of Axis is stored in Latch 0/1
//////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKCodRef (USHORT Axis, BOOL* pStatus, double* pData)
{
	double	SecondCRef;			// Value of second ref. mark during coded REF
	double	OffsetCRef;			// Coded offset value from coded REF

	*pStatus = FALSE;

	if (Axis>15) return FALSE;

	switch (CodRefSta[Axis])
	{
		case 1:					// Wait for 1st reference mark
				if (!IKLatched (Axis, CodRefLat[Axis], &RefStatus)) return FALSE;
				if (RefStatus)
				{
					if (!IKRead48 (Axis, CodRefLat[Axis], &FirstCRef[Axis])) return FALSE;	// Read value of first ref. mark
					if (!IKLatchREF(Axis, CodRefLat[Axis])) return FALSE;					// Latch with next ref. mark
					CodRefSta[Axis]=2;		// Set coded ref. status to wait for 2nd ref. mark
				}
				break;

		case 2:					// Wait for 2nd reference mark
				if (!IKLatched (Axis, CodRefLat[Axis], &RefStatus)) return FALSE;
				if (RefStatus)
				{
					if (!IKRead48 (Axis, CodRefLat[Axis], &SecondCRef)) return FALSE;		// Read value of second ref. mark
					if (SecondCRef!=FirstCRef[Axis])							// Different ref. mark traversed
					{
						OffsetCRef = (SecondCRef-FirstCRef[Axis])/1024;
						if (fabs(OffsetCRef)>=CodRefDist[Axis])
						{
							CodRefSta [Axis] = 0;	// Stop coded REF
							return FALSE;			// Wrong REF distance
						}

						if (OffsetCRef>0)			// Positive direction
						{
							if (OffsetCRef>(CodRefDist[Axis]/2))
								OffsetCRef=(OffsetCRef-(CodRefDist[Axis]/2+1))*CodRefDist[Axis];
							else
								OffsetCRef=((CodRefDist[Axis]/2)-OffsetCRef)*CodRefDist[Axis]-OffsetCRef;
						}
						else						// Negative Direction
						{
							OffsetCRef=fabs(OffsetCRef);
							if (OffsetCRef>(CodRefDist[Axis]/2))
								OffsetCRef=(OffsetCRef-(CodRefDist[Axis]/2+1))*CodRefDist[Axis]+OffsetCRef;
							else
								OffsetCRef=((CodRefDist[Axis]/2)-OffsetCRef)*CodRefDist[Axis];
						}
						*pData=OffsetCRef*1024-FirstCRef[Axis];  *pStatus=TRUE;
						CodRefSta[Axis]=0;
					}
					else												// Same reference mark traversed
					{
						if (!IKLatchREF(Axis, CodRefLat[Axis])) return FALSE;		// Latch again with next ref. mark
					}

				}
				break;

		default:
				return FALSE;
	}
	return TRUE;
}


/////////////////////////////////////////////////////////////////////
// IKWaitLatch: Wait until count value of Axis is stored in Latch 0/1
/////////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKWaitCodRef (USHORT Axis, double* pData)
{
	double	SecondCRef;			// Value of second REF mark during coded REF
	double	OffsetCRef;			// Coded offset value from coded REF
	
	if (Axis>15) return FALSE;
	if (CodRefSta[Axis]==0) return FALSE;		// Coded REF not started

  // Wait for 1st reference mark
	do
	{
		if (!IKInputW ( Axis, ContrReg, &RegW )) return FALSE;
	} while ( !(RegW & (cmdLatch0<<CodRefLat[Axis])) );

  // Read value of 1st reference mark
	if (!IKRead48 (Axis, CodRefLat[Axis], &FirstCRef[Axis])) return FALSE;	// Read value of first ref. mark
	if (!IKLatchREF(Axis, CodRefLat[Axis])) return FALSE;						// Latch with next ref. mark
	CodRefSta[Axis]=2;		// Set coded Ref status to wait for 2nd Ref mark


  // Wait for 2nd reference mark
	do
	{
		do
		{
			if (!IKInputW ( Axis, ContrReg, &RegW )) return FALSE;
		} while ( !(RegW & (cmdLatch0<<CodRefLat[Axis])) );

	  // Read value of 1st reference mark
		if (!IKRead48 (Axis, CodRefLat[Axis], &SecondCRef)) return FALSE;

		if (SecondCRef=FirstCRef[Axis])	// Same REF mark traversed
			if (!IKLatchREF(Axis, CodRefLat[Axis])) return FALSE;		// Latch again with next ref. mark

  // Read value of 1st reference mark
	} while (SecondCRef!=FirstCRef[Axis]);	// Wait until different reference mark is traversed


	OffsetCRef = (SecondCRef-FirstCRef[Axis])/1024;
	if (OffsetCRef>=CodRefDist[Axis])
	{
		CodRefSta [Axis] = 0;	// Stop coded REF
		return FALSE;			// Wrong REF distance
	}

	if (OffsetCRef>0)			// Positive direction
	{
		if (OffsetCRef>(CodRefDist[Axis]/2))
			OffsetCRef=(OffsetCRef-(CodRefDist[Axis]/2+1))*CodRefDist[Axis];
		else
			OffsetCRef=((CodRefDist[Axis]/2)-OffsetCRef)*CodRefDist[Axis]-OffsetCRef;
	}
	else						// Negative direction
	{
		OffsetCRef=fabs(OffsetCRef);
		if (OffsetCRef>(CodRefDist[Axis]/2))
			OffsetCRef=(OffsetCRef-(CodRefDist[Axis]/2+1))*CodRefDist[Axis]+OffsetCRef;
		else
			OffsetCRef=((CodRefDist[Axis]/2)-OffsetCRef)*CodRefDist[Axis];
	}

	*pData=OffsetCRef*1024-FirstCRef[Axis];
	CodRefSta[Axis]=0;

	return TRUE;
}


/////////////////////////////////////////////////////
// IKStopCodRef: Stops REF with coded reference marks
/////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKStopCodRef (USHORT Axis)
{
	if (Axis>15) return FALSE;
	CodRefSta [Axis] = 0;
	if (!IKInputW(Axis,RefCmdReg,&RegW)) return FALSE;
	return IKOutput(Axis, RefCmdReg, (USHORT)(RegW & ~cmdLatchREF));
}


/////////////////////////////////////////////////////////////////////
// IKClear: Clears error bits in Axis (frequency and amplitude error)
/////////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKClear (USHORT Axis)
{
	return IKOutput ( Axis, ContrReg, cmdClear );
}


////////////////////////////////////////////////////
// IKStatus: Reads counter status of Axis to pStatus
////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKStatus (USHORT Axis, ULONG* pStatus)
{
	*pStatus = 0;
	if (!IKInputW ( Axis, ContrReg, &RegW )) return FALSE;
	*pStatus=(ULONG)(RegW & 0x0153);		// Latch 0/1, counter start/stop, frequency error

	if (!IKOutput ( Axis, StatReg3, cmdADFreeze ));		// Freeze value
	if (!IKInputW ( Axis, RefCmdReg, &RegW )) return FALSE;
	if (!IKOutput ( Axis, StatReg3, cmdADFree ));		// Unfreeze value
	*pStatus=*pStatus | ( (ULONG)(RegW & 0x3F3F)<<9 );	 // REF, Amplitude

	if (!IKInputW ( Axis, StatReg3, &RegW )) return FALSE;
	*pStatus=*pStatus | ( (ULONG)(RegW & 0xFF00)<<16 );	 // IC code

	return TRUE;
}


///////////////////////////////////////////////////////////////////////
// IKRead32: Reads 32-bit count value of Axis from Latch 0/1 to pBuffer
///////////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKRead32 (USHORT Axis, USHORT Latch, LONG* pData)
{
	if (Latch>1) return FALSE;
	return IKInputL ( Axis, (USHORT)(Latch*6), pData);
}


///////////////////////////////////////////////////////////////////////
// IKRead48: Reads 48-bit count value of Axis from Latch 0/1 to pBuffer
///////////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKRead48 (USHORT Axis, USHORT Latch, double* pData)
{
	__int64	Count=0;
	USHORT* pCount = (USHORT*)&Count;

	if (Latch>1) return FALSE;

	if (!IKInputL ( Axis, (USHORT)(Latch*6), (ULONG*)pCount) ) return FALSE;

	pCount = pCount+2;
	if (!IKInputW ( Axis, (USHORT)(Latch*6+4), pCount) ) return FALSE;

	if (*pCount & 0x8000)
	{
		pCount=pCount+1;
	    *pCount=0xFFFF;
	}
	*pData=(double) Count;
	return TRUE;
}
